/**
 * \file      test-mac-hmac-sha1.c
 *
 * \brief     The HMAC-SHA-1 message authentication code unit-tests
 *
 * \warning   SHA-1 is considered a weak message digest and its use constitutes a
 *            security risk! We recommend considering stronger message digests
 *            instead!
 *
 * \copyright 2022 Joseph Benden <joe@benden.us>
 *
 * \license   GPL-2.0-OR-LATER
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>

#include "aircrack-ng/crypto/crypto.h"
#include "tests.h"

// clang-format off
/*
 * RFC 2202 test vectors
 */
struct {
	uint8_t         key[96];
	size_t          key_len;
	uint8_t         data[96];
	size_t          data_len;
	uint8_t         digest[DIGEST_SHA1_MAC_LEN];
} sha1_tests[] = {
	{
		{0x0b, 0x0b, 0x0b, 0x0b,
		 0x0b, 0x0b, 0x0b, 0x0b,
		 0x0b, 0x0b, 0x0b, 0x0b,
		 0x0b, 0x0b, 0x0b, 0x0b,
		 0x0b, 0x0b, 0x0b, 0x0b},
		20,
		"Hi There",
		8,
		{0xb6, 0x17, 0x31, 0x86,
		 0x55, 0x05, 0x72, 0x64,
		 0xe2, 0x8b, 0xc0, 0xb6,
		 0xfb, 0x37, 0x8c, 0x8e,
		 0xf1, 0x46, 0xbe, 0x00}
	},
	{
		"Jefe",
		4,
		"what do ya want for nothing?",
		28,
		{0xef, 0xfc, 0xdf, 0x6a,
		 0xe5, 0xeb, 0x2f, 0xa2,
		 0xd2, 0x74, 0x16, 0xd5,
		 0xf1, 0x84, 0xdf, 0x9c,
		 0x25, 0x9a, 0x7c, 0x79}
#ifdef EXPENSIVE_TESTS
	},
	{
		{0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa,
                 0xaa, 0xaa, 0xaa, 0xaa,
                 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa},
		20,
		{0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
		 0xdd, 0xdd},
		50,
		{0x12, 0x5d, 0x73, 0x42, 0xb9, 0xac, 0x11, 0xcd,
		 0x91, 0xa3, 0x9a, 0xf4, 0x8a, 0xa1, 0x7b, 0x4f,
		 0x63, 0xf1, 0x75, 0xd3}
	},
	{
		{0x01, 0x02, 0x03, 0x04,
		 0x05, 0x06, 0x07, 0x08,
		 0x09, 0x0a, 0x0b, 0x0c,
		 0x0d, 0x0e, 0x0f, 0x10,
		 0x11, 0x12, 0x13, 0x14,
		 0x15, 0x16, 0x17, 0x18,
		 0x19},
		25,
		{0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
		 0xcd, 0xcd},
		50,
		{0x4c, 0x90, 0x07, 0xf4, 0x02, 0x62, 0x50, 0xc6,
		 0xbc, 0x84, 0x14, 0xf9, 0xbf, 0x50, 0xc8, 0x6c,
		 0x2d, 0x72, 0x35, 0xda}
	},
	{
		{0x0c, 0x0c, 0x0c, 0x0c,
		 0x0c, 0x0c, 0x0c, 0x0c,
		 0x0c, 0x0c, 0x0c, 0x0c,
		 0x0c, 0x0c, 0x0c, 0x0c,
		 0x0c, 0x0c, 0x0c, 0x0c},
		20,
		"Test With Truncation",
		20,
		{0x4c, 0x1a, 0x03, 0x42, 0x4b, 0x55, 0xe0, 0x7f,
		 0xe7, 0xf2, 0x7b, 0xe1, 0xd5, 0x8b, 0xb9, 0x32,
		 0x4a, 0x9a, 0x5a, 0x04}
	},
	{
		{0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //32
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //64
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa}, //80
		80,
		"Test Using Larger Than Block-Size Key - Hash Key First",
		54,
		{0xaa, 0x4a, 0xe5, 0xe1, 0x52, 0x72, 0xd0, 0x0e,
		 0x95, 0x70, 0x56, 0x37, 0xce, 0x8a, 0x3b, 0x55,
		 0xed, 0x40, 0x21, 0x12}
	},
	{
		{0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //32
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, //64
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
		 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa}, //80
		80,
		"Test Using Larger Than Block-Size Key and "
		"Larger Than One Block-Size Data",
		73,
		{0xe8, 0xe9, 0x9d, 0x0f, 0x45, 0x23, 0x7d, 0x78,
		 0x6d, 0x6b, 0xba, 0xa7, 0x96, 0x5c, 0x78, 0x08,
		 0xbb, 0xff, 0x1a, 0x91}
#endif
	}
};
// clang-format on

STATIC_ASSERT(ArrayCount(sha1_tests) >= 2u, ensure_at_least_2_entries);

int main(int argc, char ** argv)
{
	size_t i;
	int error = 0;
	uint8_t sha1sum[DIGEST_SHA1_MAC_LEN];

	(void) argc;

	for (i = 0; i < ArrayCount(sha1_tests); i++)
	{
		error |= MAC_HMAC_SHA1(sha1_tests[i].key_len,
							   sha1_tests[i].key,
							   sha1_tests[i].data_len,
							   sha1_tests[i].data,
							   sha1sum);
		error |= test(
			sha1sum, sha1_tests[i].digest, DIGEST_SHA1_MAC_LEN, argv[0]);
	}

	for (i = 0; i < ArrayCount(sha1_tests); i++)
	{
		error |= MAC_HMAC_SHA1_Vector(
			sha1_tests[i].key_len,
			sha1_tests[i].key,
			1,
			(const uint8_t * []){(uint8_t *) &sha1_tests[i].data},
			&sha1_tests[i].data_len,
			sha1sum);
		error |= test(
			sha1sum, sha1_tests[i].digest, DIGEST_SHA1_MAC_LEN, argv[0]);
	}

	return error;
}
