#include "tests/virtual_network_helpers.h"

namespace tests {
namespace virtual_network_helpers {

ssf::layer::LayerParameters GetServerTLSParametersAsFile() {
  return {{"ca_file", "./certs/trusted/ca.crt"},
          {"crt_file", "./certs/server.crt"},
          {"key_file", "./certs/server.key"},
          {"dhparam_file", "./certs/dh4096.pem"}};
}

ssf::layer::LayerParameters GetClientTLSParametersAsFile() {
  return {{"ca_file", "./certs/trusted/ca.crt"},
          {"crt_file", "./certs/certificate.crt"},
          {"key_file", "./certs/private.key"}};
}

ssf::layer::LayerParameters GetDefaultServerParameters() {
  return {{"default", "true"}};
}

ssf::layer::LayerParameters GetServerTLSParametersAsBuffer() {
  return {
      {"ca_buffer",
       "-----BEGIN "
       "CERTIFICATE-----"
       "\nMIIF3TCCA8WgAwIBAgIJAPrG1z5hkxBBMA0GCSqGSIb3DQEBBQUAMFIxCzAJBgNVBAYTA"
       "kJFMRMwEQYDVQQIEwpTb21lLVN0YXRlMSEwHwYDVQQKExhJbnRlcm5ldCBXaWRnaXRzIFB0"
       "eSBMdGQxCzAJBgNVBAMTAkNBMB4XDTE1MDYwMTAxMTgwMVoXDTI1MDUyOTAxMTgwMVowUjE"
       "LMAkGA1UEBhMCQkUxEzARBgNVBAgTClNvbWUtU3RhdGUxITAfBgNVBAoTGEludGVybmV0IF"
       "dpZGdpdHMgUHR5IEx0ZDELMAkGA1UEAxMCQ0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwg"
       "gIKAoICAQDmT2S1frBI4532bASwbuNHgK5dlSv9PbpvEPTL8G4t4SMBTNXSMr2ScXZC0SRp"
       "pTgrEyvK2P85YKaUucnX4T7Rr2i36j+UPC+509Oh+J9yp9TdBulNgM0VNiJCns+"
       "UFC2dRHVR7/0UX850YUhazISFDeC8cnrDs4VZiImeDDGlFDUFfoHU7IX8ta/"
       "NWFc6cYHoDk2/Ye+FBJ1RM4iw7n5DjW8sdTM886GaCjlyLt2R0iY9lVQnuAuMn/"
       "TzWCE86TWfpgYZz2uiBIs7ZhOckv372Mij7D12kKG882StT2jqihMz1ncCkKquWdQoXZs+"
       "sTQfvlS9SKCTvIog4aMbBvIMfzUkFUJMxAk0CBe49OFOdfNw7eTES7TuI90D08WB5cPBOrB"
       "LS66xAdw7DFIT49dxA9tQTqzgp5wfyQUW6RhNiygNeamIY7HwPFQsFC7WKwIfYcXQWQAZDS"
       "0YdSXSvrqS2paoilVfxLsi1SM2NdiQudUIbTPYAblAlP7Mbe90QizzeDbLgEJw5Anapyksw"
       "2SPwyLSVM5uXFy/KS9fYAyBmSr0XEaTS7SC76uHPJa4Iffv+gKzJkB/UWNbz39PSF+RXuz/"
       "Dvc7DskOPnkZcm4KZtOiS/dzB/"
       "B8+"
       "0EWhFn6n2mIlq2eGwwXf62XFeUiQ0wozrzQRWoSUF6XiqZnU6SZ7wIDAQABo4G1MIGyMB0G"
       "A1UdDgQWBBRRdZlEXKYtmKsW5HmAxoSyKH/"
       "YMjCBggYDVR0jBHsweYAUUXWZRFymLZirFuR5gMaEsih/"
       "2DKhVqRUMFIxCzAJBgNVBAYTAkJFMRMwEQYDVQQIEwpTb21lLVN0YXRlMSEwHwYDVQQKExh"
       "JbnRlcm5ldCBXaWRnaXRzIFB0eSBMdGQxCzAJBgNVBAMTAkNBggkA+"
       "sbXPmGTEEEwDAYDVR0TBAUwAwEB/"
       "zANBgkqhkiG9w0BAQUFAAOCAgEAmDjjNo2bLDoFeMhLGcV61F4ZdB1wWJvfufGmFjHGjXTl"
       "gXA5u/"
       "L+fz1LqLqMoQHRxUczL0ZgE+"
       "8JwXaWsqdYWnz8F1h6u0HZemWb5YlaNDeHCrTcb4Xi8SS6A0AbEdL2M6ydOOxtjtDCTMxrT"
       "AAPmDVJN9Kfl7+"
       "kmgjuAWbVdeRSP0sKVqocCGmENX61tWaT9zappUb9l6wxe7J48geubN8Q6hAzarrqdQFvNG"
       "8DkZkizHKXwGVO0v9SuFhRy8nGb41OCHiRbLunzkQbJfwCWOL9kfH/"
       "u9R11Eli8Wq2A0pEOb7zx5m1B+RRhlBtOpXgsRpJaGrNBrj6ATa2gc0XvU4f9aIr/"
       "urHKu1FO/q1K4D+yelS7E87HohXdMmnhKNYl3L/"
       "Y1ylQnDVwXb3uGK5HeWIf0EvcgPPQdabReQgh2Ybn40Ec5txtlkfQXQ76VBOosEZ+"
       "Kbep50J1szt5wrJe5MqhZIxvTSVwW3iwt2v2VYLzGX+"
       "zCcPaXFCVlEelLVF4VGXOxzm4pMmoBWe/"
       "0+BJJkQoO0B8PWaDHIjxQTCHo8enBf1yWUOTBysuL+HA8rZ0VC5OOVvHWeqS5C3jlYalph+"
       "bp6iO/FAoIeigorNWkDQrmNqH6F+37i9Jse8vrPsbtxA1YwYfKeIDxO6CMHH1aDJ/"
       "d7Oqd2Bda1uHMKMm4Y=\n-----END CERTIFICATE-----"},
      {"crt_buffer",
       "-----BEGIN "
       "CERTIFICATE-----\nMIIFRTCCAy2gAwIBAgIJAK8e45jVbgd/"
       "MA0GCSqGSIb3DQEBBQUAMFIxCzAJBgNVBAYTAkJFMRMwEQYDVQQIEwpTb21lLVN0YXRlMSE"
       "wHwYDVQQKExhJbnRlcm5ldCBXaWRnaXRzIFB0eSBMdGQxCzAJBgNVBAMTAkNBMB4XDTE3MD"
       "EyNzEzMTYzN1oXDTI3MDEyNTEzMTYzN1owVjELMAkGA1UEBhMCQkUxEzARBgNVBAgMClNvb"
       "WUtU3RhdGUxITAfBgNVBAoMGEludGVybmV0IFdpZGdpdHMgUHR5IEx0ZDEPMA0GA1UEAwwG"
       "U2VydmVyMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAvk2u8gG75eSrrtmNYy9"
       "y52MFSLo81X322cs3m5jHT4N2Rm0ouGCsetD2zOhMkXUtsaWEIflVqc7Js1BLskxHcHerYg"
       "xkqBjs3sEWAS71xlbksdAexLw7l32a5WSAXKUl01MCROJ0R+Yb2xP5crG/nZcPB/"
       "zHhcyaQqEMZHKtp9180g13Qr9B+"
       "2fVp8FgHeGtF4NgXsuilPckny6g6gac8aiy0pJ50KAmdlBwPuax8/"
       "gvMLEQhF6aHV9h2rXLg+d+QbScctRz4mm2X4siQJczV4+DsVyp8NqPNgpU+ht/"
       "X3VpDm4X+kohAOJE+"
       "z9oZ4G5CMw7Mdo8ZgFkvly16LtFhxVmHFVPla3PtOFxgo4npDVu9IKAYVEJYQ6w3GSTRaXn"
       "Wm/"
       "TKSQPQWQEGnLOGltWjCXrcjkprrRmb3fBjpwEGCDXcDxtGc3A0hWIoyhktyHthCyoeEuyTM"
       "oUzyUBMndZCyd4sJPUjK+yHuO/vlEMe14aVmKpMfv3+of/"
       "dgvBlM6Oqx920tCnnB7paupUGNNxC8BNyaOj76P7DEmwJ+"
       "ZjeJwfO04jM0QZyS0CIXoDPsmkWRnBmNNxqfUNOA4kjeXZ0krlhnIeUODEiZ06wg/"
       "yBK6Q98Y1c+JR9hlKtWjUEawp/"
       "ZAfzR2N12zx9hgzsTkuWmACmBSU5rMLuntoSTqtBx8CAwEAAaMaMBgwCQYDVR0TBAIwADAL"
       "BgNVHQ8EBAMCBeAwDQYJKoZIhvcNAQEFBQADggIBAJiJy9To/IWujFM9oSMWv0kDMHjj/"
       "VRsSfh8ayYcHonIY0uom1FhKc1jXSFJywQZIiSnKDIaOVls50FOeLHeQQJwR0JzeUGx3gRk"
       "h4wTBJV/"
       "nNd1OzQC0XPsKWDPhIT59ADOBL3F9aIYpINmENrjHyqesM8YiSH9hgkx4ZPqEdYqGOAYr13"
       "8DldOXMyotxcYA1mds/"
       "ZLG1flmi0WkyoOSegzpxIsY0OAfMDz0Q+"
       "1XOmuxD76SofigkccDMXq1GZBU8ZppVRoLiZkLu0gEFZxLyTTlAC9ZWZSPczrLW0T6tu70l"
       "m3VKTeHDwmMoTfjv5zpSS8zFf3lE9AEtK/"
       "r1HmiwAV8mB1vsRfULaKHcjbclJswiB1hrW3Sn0bpWfwuSS/"
       "6CO7L23NjKhM41fxd6qjlDJFYuxaOcQZ9K9Vs+"
       "BrWRGoDXo1GdVqRk1RbZDVGNiSNuM43RzKCMjsIA4P7in71rtGHvIexBTp7q2dAyMgDHlKT"
       "y+bVc/NyBdYjfPI87vB/"
       "SNgFPPwlJl5N8fBelbl1lO0A1I6kVulRn5XD4wXW2R2KE3uIJWf7vsdOEPN0GP+"
       "OO7h0S5STHkasIXnM9PUxA1Ytrg+xX2hN7vYM4AEvTl2cYOZLc5/"
       "AVo+KgQV8zBohrB+hw4L1ksKz6+XQFEHKqIi0i3aWNDKq4SabXH0i9kd\n-----END "
       "CERTIFICATE-----"},
      {"key_buffer",
       "-----BEGIN PRIVATE "
       "KEY-----\nMIIJQwIBADANBgkqhkiG9w0BAQEFAASCCS0wggkpAgEAAoICAQC+"
       "Ta7yAbvl5Kuu2Y1jL3LnYwVIujzVffbZyzebmMdPg3ZGbSi4YKx60PbM6EyRdS2xpYQh+"
       "VWpzsmzUEuyTEdwd6tiDGSoGOzewRYBLvXGVuSx0B7EvDuXfZrlZIBcpSXTUwJE4nRH5hvb"
       "E/lysb+dlw8H/"
       "MeFzJpCoQxkcq2n3XzSDXdCv0H7Z9WnwWAd4a0Xg2Bey6KU9ySfLqDqBpzxqLLSknnQoCZ2"
       "UHA+5rHz+"
       "C8wsRCEXpodX2HatcuD535BtJxy1HPiabZfiyJAlzNXj4OxXKnw2o82ClT6G39fdWkObhf6"
       "SiEA4kT7P2hngbkIzDsx2jxmAWS+XLXou0WHFWYcVU+Vrc+"
       "04XGCjiekNW70goBhUQlhDrDcZJNFpedab9MpJA9BZAQacs4aW1aMJetyOSmutGZvd8GOnA"
       "QYINdwPG0ZzcDSFYijKGS3Ie2ELKh4S7JMyhTPJQEyd1kLJ3iwk9SMr7Ie47++"
       "UQx7XhpWYqkx+/f6h/92C8GUzo6rH3bS0KecHulq6lQY03ELwE3Jo6Pvo/"
       "sMSbAn5mN4nB87TiMzRBnJLQIhegM+"
       "yaRZGcGY03Gp9Q04DiSN5dnSSuWGch5Q4MSJnTrCD/IErpD3xjVz4lH2GUq1aNQRrCn9kB/"
       "NHY3XbPH2GDOxOS5aYAKYFJTmswu6e2hJOq0HHwIDAQABAoICAHYWSFe9wbXfFrWyI60HdX"
       "qllFLT1gn4VyRAmOoBfLQXdF4xQFJhEsZHQs5V26VrHAafGYXnbUOooLqXOdGSycW9TGjUH"
       "eW9ScbT8+"
       "6aEEKaCuFXrrAPyb6wX1y253wVsooWeVL7XoJ7BJAGLgUAzVFhuh8PuVOKabKssr7l9mIy6"
       "1CAsJ+HhM8XWJn3QchpEHqFuFg6blGXpm7shUeTAPJn50cJQ3O03QkP0wcFKimkuMuzI/"
       "evqFxPoq4PnxDjphHpnCcZNCurw1I4SPHhoSdVU7fMM1vUk4dKE/"
       "767yfx+FXO9SwHscEVpw5tA0EvWv1dleHSdSYq+"
       "TOy3eJjyrj77iLPzJkbIIQ9mMZDzWqMT8+"
       "yFv0FE3v3YULzbiAHf8koWbXWtTHw197JXPIQ+qLSdl0NGPimICIyf9W7fcvh+ANihCELF+"
       "+N6Gw7wbbIHJnnEaw6GkD2+smGDfQLUMVrCEtrSZujX+/"
       "R9MNNdRn5sSnlNOSLoNqUjgFHW+"
       "vl1xXJf4wWtOlw2GjhQMZ5GWn08Byiem0shPN6z8mukoes/"
       "9zPxy2NPMnYGMCEyf+"
       "RGQB3la33WIkbqH5SMZd1KRRlVn2amKrrxADCM5g2pmpUFJApT3jX2hPxinRpvlAPDpe+"
       "prr/"
       "sSMArUC9a22iCz7OusDaZgSznJD3KivHaAURAoIBAQDyPoJ5cQ+nhujpxbhm5KdaGHi/"
       "8DCfS8qmcX/"
       "hbZo8LdR0dwh1VrLllGoJ4KPAQZPIajLtJgSfpUWcSuhws4S4wnNinJZM4vxA+"
       "9pBVJUEdJlO7ptckS8m0JtVF2B09Aahd4yCz8oEiWRDAJumPrXHh5CEMTLszzhQqc38u2Ev"
       "1tW+c0TnG7g64FlEXyGxRh6tPeg91BvIw5pcdxoJd96IbjmsR3UaO+Ulr+/"
       "TE3IYD+E6Bnrt5HPH7cyOGrndw9KPsuTP0HkS/4naAblC1MiKV7fdzDW2/"
       "cRHQdL6UW6ApuDyf7Mz5JgLkivtLZZ7qG0DFMwr2hWwiyrffCCVNvlnAoIBAQDJHB08ju4F"
       "Ko2YeKF9eV/kVMcGJg/FWK8o/"
       "lz0c298lfVnLpaV9l0KNls2InozYsC5DWWGFFxC5Garac5HcaQovFn4SrP3qQJ07Bptq4Qy"
       "96r0an5QnhJbCFImHuyy1tPDbUfGRwws/"
       "yyedtKhfA9CnZpcmoUTvutDcud8MP5fMFKfB8mGTidmc1SuDFWQ5eG/"
       "W0j3U84YwYoDT3lUV43ETSl+oaZ0LwoaBrGblv5MQ5+"
       "5Xjrqp1oeolNA8AO3zu5MNFtJV0t8Hf6yi4c0Mpmzj+lr1bLvCRvYzj8f7/"
       "1WcK5hxO3qA0ZUzSguG3tnYfoq0SGvKZogMBghjw/"
       "aVZmJAoIBAQDxLQr5nXbYc5iwJgnRZxS2vmZ5sci2z96Sl62Z4mUMRRSgX2IscITrzTdFXk"
       "Gpx6A6iIDrWk6AIyuKz4lbq18YE4CNTRjWs6YxjI1fPbafGPcJfsVvA5zOKCJ+"
       "NSfzSIDRgRSDqByZJ7CKhLByczzHylA3TERHQIfwxx4t4OCBoIJtepW/"
       "AfoJpBM73vrMY+"
       "Nm15vA8sw2tYeZR3rEsnL7ocIJo5sS407glqlOnK2bBzRVcN4yuqALDNX7JOCnya55IsJrp"
       "ELmh0PXqJ+pye9ezLZiPE2+"
       "QiiISSCj1y8NJxgOVtUzdQNrgbv6BskrlCP0TWB4yEeP3P5yQkKuTG1BAoIBAQDClTMsjtZ"
       "5pSyPY9hMgQ1o7SYMKvODbEU+"
       "Eq55zt8F2kow5YDJ5C8tXF5wSHeYFGqIIhAi5LxH9JnqLFtsAOZmi6c5d34dmCdKXc9MYXF"
       "tpvh/"
       "uDIzal40Oh7Rx4YZm4YVRjz1+XQsbVEMmcc+"
       "e7yAJeFezBsKeeVDw396SCU2kJjASlaQXjn9BIXgSQrZQbNk5Lv6odl9VnVkspagJn6Cem4"
       "1mc5mB8VsXSGnezbdL2jxf9zdgcLOdRy8bf4K8GrczKOumInus07mDd5pgLP6arncnk2cvm"
       "F1Kkkla5zg7/Gpo1LW4wg/"
       "sKi5w9lotAHLP8Gpq0T289HFVphuFfiRAoIBAFEP4pP5oIDRpAKaY037H7a/"
       "AVe5RyecdchxbK3fEMUNLdQ30MTs4WIy1Q6mR/Mt2nxFOMU7QC6Yfc79GqdqP6I/"
       "8GeLolyda2j2phJ1y1JGcgQjeGpPSkrsJzvMfwrscn0+"
       "hmvdeIAmDyH0GCjr5SbQb5CTYZp9Ds3Ay8c1CDFgfdo+qoFRxc0F93x/"
       "TekGXWxa9b5ixEFJc0K7Ty/X1lcOcfWSu3oUwZ6jNryq6loGhUM3aDkrxyeWrKiP0IuMT/"
       "pWK6xS/"
       "xkfqFLUqzHaNaQ8JtoJxdXsUEnPWhCrDbLDtiMB4UbW7iu5jjuaLlsSaqr8RdfwkFOJj43w"
       "zeb9RQM=\n-----END PRIVATE KEY-----"},
      {"dhparam_buffer",
       "-----BEGIN DH "
       "PARAMETERS-----\nMIICCAKCAgEAseqOk+DzWryjmv42VdCyQ+"
       "7xOyzM5hq9fLpAuM5OWzl0ieTuKs7DoDbYiWW61FwPupMThl1O7Sp6N2eFqsppnRDuR9ZGM"
       "HbEjJQCHwwQzWlXyB1u5xv9IOIxIZPXiR7IT22xGiB2YjS7oMZguRaAQWys8SG7N5eKeQ30"
       "G8FS+DZmS4wEGUacAh1qysta4pzyWeQNUDgwghyT+"
       "EsTrdTUuWt6ebbnIHnKbr3FQVs6SVZlp2IEun3V4WrTs19tt2nwxn4Ttig+9/"
       "WHePgmxYzMM3KImJNEOO3iDwl7w0sJp3j5RGW5Dh/"
       "CvVNtlbo1wZIWLVZTFBTlwVFVCMSRMoghwLoG5hC4GyQVT8rdj+zn0+"
       "VLXKCqyDXge3YzqnbMBeI1sp2ai4RFjd+d0jt1mXOaGXNV4dYSyQh6+7AFzptDwsSZ7/"
       "kfr/SYw0xtMu/vqcj8TnQO8Lq3s2Et1oF+MFbh/"
       "kWCagYLcXG1eD8Lmxi104NwRMQYTsUzIRcjLN5xdI2CtQVIbg1zdC3+Ciu+"
       "xJXfg1fYK9ZLBst31Pj/ZUaJlR0IOumboE4kobGA87LOAbXB5LfTj7hROOhn4/NJLj/"
       "CHXXCPpXixE890Z5wo2f/PG+4umL4DTkKhr/"
       "xGr7EJr1iLuv8ulDaApPR4MYbfEOUwpo3oqBRJFh1m5gahndmtqMCAQI=\n-----END DH "
       "PARAMETERS-----"}};
}

ssf::layer::LayerParameters GetClientTLSParametersAsBuffer() {
  return {{"ca_buffer",
           "-----BEGIN "
           "CERTIFICATE-----\n"
           "MIIF3TCCA8WgAwIBAgIJAPrG1z5hkxBBMA0GCSqGSIb3DQEBBQUAMFIxCzAJBgNVBAY"
           "TAkJ"
           "FMRMwEQYDVQQIEwpTb21lLVN0YXRlMSEwHwYDVQQKExhJbnRlcm5ldCBXaWRnaXRzIF"
           "B0eS"
           "BMdGQxCzAJBgNVBAMTAkNBMB4XDTE1MDYwMTAxMTgwMVoXDTI1MDUyOTAxMTgwMVowU"
           "jELM"
           "AkGA1UEBhMCQkUxEzARBgNVBAgTClNvbWUtU3RhdGUxITAfBgNVBAoTGEludGVybmV0"
           "IFdp"
           "ZGdpdHMgUHR5IEx0ZDELMAkGA1UEAxMCQ0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwA"
           "wggI"
           "KAoICAQDmT2S1frBI4532bASwbuNHgK5dlSv9PbpvEPTL8G4t4SMBTNXSMr2ScXZC0S"
           "RppT"
           "grEyvK2P85YKaUucnX4T7Rr2i36j+UPC+509Oh+J9yp9TdBulNgM0VNiJCns+"
           "UFC2dRHVR7/0UX850YUhazISFDeC8cnrDs4VZiImeDDGlFDUFfoHU7IX8ta/"
           "NWFc6cYHoDk2/Ye+FBJ1RM4iw7n5DjW8sdTM886GaCjlyLt2R0iY9lVQnuAuMn/"
           "TzWCE86TWfpgYZz2uiBIs7ZhOckv372Mij7D12kKG882StT2jqihMz1ncCkKquWdQoX"
           "Zs+"
           "sTQfvlS9SKCTvIog4aMbBvIMfzUkFUJMxAk0CBe49OFOdfNw7eTES7TuI90D08WB5cP"
           "BOrB"
           "LS66xAdw7DFIT49dxA9tQTqzgp5wfyQUW6RhNiygNeamIY7HwPFQsFC7WKwIfYcXQWQ"
           "AZDS"
           "0YdSXSvrqS2paoilVfxLsi1SM2NdiQudUIbTPYAblAlP7Mbe90QizzeDbLgEJw5Anap"
           "yksw"
           "2SPwyLSVM5uXFy/KS9fYAyBmSr0XEaTS7SC76uHPJa4Iffv+gKzJkB/"
           "UWNbz39PSF+RXuz/"
           "Dvc7DskOPnkZcm4KZtOiS/dzB/"
           "B8+"
           "0EWhFn6n2mIlq2eGwwXf62XFeUiQ0wozrzQRWoSUF6XiqZnU6SZ7wIDAQABo4G1MIGy"
           "MB0G"
           "A1UdDgQWBBRRdZlEXKYtmKsW5HmAxoSyKH/"
           "YMjCBggYDVR0jBHsweYAUUXWZRFymLZirFuR5gMaEsih/"
           "2DKhVqRUMFIxCzAJBgNVBAYTAkJFMRMwEQYDVQQIEwpTb21lLVN0YXRlMSEwHwYDVQQ"
           "KExh"
           "JbnRlcm5ldCBXaWRnaXRzIFB0eSBMdGQxCzAJBgNVBAMTAkNBggkA+"
           "sbXPmGTEEEwDAYDVR0TBAUwAwEB/"
           "zANBgkqhkiG9w0BAQUFAAOCAgEAmDjjNo2bLDoFeMhLGcV61F4ZdB1wWJvfufGmFjHG"
           "jXTl"
           "gXA5u/"
           "L+fz1LqLqMoQHRxUczL0ZgE+"
           "8JwXaWsqdYWnz8F1h6u0HZemWb5YlaNDeHCrTcb4Xi8SS6A0AbEdL2M6ydOOxtjtDCT"
           "MxrT"
           "AAPmDVJN9Kfl7+"
           "kmgjuAWbVdeRSP0sKVqocCGmENX61tWaT9zappUb9l6wxe7J48geubN8Q6hAzarrqdQ"
           "FvNG"
           "8DkZkizHKXwGVO0v9SuFhRy8nGb41OCHiRbLunzkQbJfwCWOL9kfH/"
           "u9R11Eli8Wq2A0pEOb7zx5m1B+RRhlBtOpXgsRpJaGrNBrj6ATa2gc0XvU4f9aIr/"
           "urHKu1FO/q1K4D+yelS7E87HohXdMmnhKNYl3L/"
           "Y1ylQnDVwXb3uGK5HeWIf0EvcgPPQdabReQgh2Ybn40Ec5txtlkfQXQ76VBOosEZ+"
           "Kbep50J1szt5wrJe5MqhZIxvTSVwW3iwt2v2VYLzGX+"
           "zCcPaXFCVlEelLVF4VGXOxzm4pMmoBWe/"
           "0+BJJkQoO0B8PWaDHIjxQTCHo8enBf1yWUOTBysuL+"
           "HA8rZ0VC5OOVvHWeqS5C3jlYalph+"
           "bp6iO/FAoIeigorNWkDQrmNqH6F+37i9Jse8vrPsbtxA1YwYfKeIDxO6CMHH1aDJ/"
           "d7Oqd2Bda1uHMKMm4Y=\n-----END CERTIFICATE-----"},
          {"crt_buffer",
           "-----BEGIN "
           "CERTIFICATE-----\nMIIFQzCCAyugAwIBAgIJAK8e45jVbgd+"
           "MA0GCSqGSIb3DQEBBQUAMFIxCzAJBgNVBAYTAkJFMRMwEQYDVQQIEwpTb21lLVN0YXR"
           "lMSE"
           "wHwYDVQQKExhJbnRlcm5ldCBXaWRnaXRzIFB0eSBMdGQxCzAJBgNVBAMTAkNBMB4XDT"
           "E1MD"
           "YwMTAxMTkwOVoXDTI1MDUyOTAxMTkwOVowVDELMAkGA1UEBhMCQkUxEzARBgNVBAgTC"
           "lNvb"
           "WUtU3RhdGUxITAfBgNVBAoTGEludGVybmV0IFdpZGdpdHMgUHR5IEx0ZDENMAsGA1UE"
           "AxME"
           "Q2VydDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBANUIAJQm7D9tLzS0U6F"
           "4hHb"
           "ZcgLjtELZJQ8jmpROYYyA9MIBDdrIbgIHeSBS/"
           "q1VRNvoDfnJBCnVcplCDMBA0uiuIGxHzO4JnyTZElWulujdoHKTfFmjxq1l46O6jb1z"
           "t1zy"
           "z92qzbOaENob2o65BZHLoPUt1UIIRLx/"
           "IGOzhh+Gmcz4gLs9PRuC7puacgwY3RCFLYyBk7R+"
           "kdUt30vQbPewTbdMmf5pGyAPBmUJpk3PpwHLiRA+qIiD3r/"
           "bR1RgaG000n57okbYpYwhHh8CmZUTZGNg0Ymb+"
           "8TXHnzMH4cC3wdwRYwdeLuNgw99k5FrH2q3g3bzTHdD56a9iA4ZRRSlkMpPXL1sCO66"
           "vPu6"
           "UB/RHZhC0hCSU/B+L+obW5UydG1o8C2Vz8h+KThzibCyuSWh87piu9MHKi7vojH/"
           "cjMSBOpUhZ0jH9xkPbTYKg4ABvdv1WcwBDAbR7c6uJw5XIsxUQ9U2aEo7D+"
           "gIkXfVwHFqlclmYCG2rmD5yhV1wvsDFCoDKksGTEVW54aiF7GimsuJo0OROTqNmBgRw"
           "hx7C"
           "KioiRg6XVjuCVb/"
           "FHYPcJxJrT5bOB+2wSa0ejzWZ7nXYSIxVMkEYKGf7DpKGSmO82vg19lHVWr7vq+"
           "1M8XimOcVRosEWVjgSfLpLZRruyHvxfGw9D2fDMQjYq7Vae5K1QLAgMBAAGjGjAYMAk"
           "GA1U"
           "dEwQCMAAwCwYDVR0PBAQDAgXgMA0GCSqGSIb3DQEBBQUAA4ICAQAm6QoPEOQV9yfotb"
           "QRKn"
           "uH8qzDW+"
           "3UcNbcMvhisrAhQPGukqTkFtenr1WK74KP2X7Y8HL7OIhwPCWvNDTW7XfrTQSFSWvIP"
           "Z/"
           "dFqflmxQ3QMiYf4iMktUZ4E3Z/eVppKBgQnJ/gEk+3pxn/"
           "7APnOisoJII2A9hc5LrbfzrsrYxaAaRUOzXPXHqStnY52uGFV8FipBx8zwchR5oeIOs"
           "m/"
           "MHtEto9a/LOsJblEia5A/7Z43/"
           "ionqaKhlwFreSGVN3lhUKDXPidm6DjUHeNRGHzSGlGSyWgCRzv9qDh1dhwBbQ6g44KS"
           "XZWR"
           "FDS+aKJ80dTVggntbaVvDIHtGpWy/xOB5b2j6BIWrUgrQzhHVhM/"
           "rnqw0iICPrk1oQxdAJsIPbUEijM0T5dbAdV8/WkxkJ3pbX/"
           "Z2F8LDfKV0W+05IELxa7ZqpuSp1pGdhPVyPo+"
           "5qbwic8xMGlK3Pf3Mq0t6m8RHq78ZaH433i37Wd4M90hB+qWItiZ+"
           "MyI7999A1baHd3bJHxRFpegCDxdbTLfYc5q0pWpIonccLsqSLDQfV1pus1vCvEnMarZ"
           "KzmL"
           "RuX2dO8LJy77qe/"
           "zTcA30pAFYLq5WkErWZU63EDWyMQcwd0vH4hljAk6C4SQPa73A+cp/"
           "ilkmrHuKkSMT0CjsbJ5EsLRBTaHKJ9N/+9c3Gm7FzklbMw==\n-----END "
           "CERTIFICATE-----"},
          {"key_buffer",
           "-----BEGIN RSA PRIVATE "
           "KEY-----"
           "\nMIIJKgIBAAKCAgEA1QgAlCbsP20vNLRToXiEdtlyAuO0QtklDyOalE5hjID0wgEN"
           "2shuAgd5IFL+rVVE2+gN+"
           "ckEKdVymUIMwEDS6K4gbEfM7gmfJNkSVa6W6N2gcpN8WaPGrWXjo7qNvXO3XPLP3ar"
           "Ns5oQ2hvajrkFkcug9S3VQghEvH8gY7OGH4aZzPiAuz09G4Lum5pyDBjdEIUtjIGTt"
           "H6R1S3fS9Bs97BNt0yZ/"
           "mkbIA8GZQmmTc+"
           "nAcuJED6oiIPev9tHVGBobTTSfnuiRtiljCEeHwKZlRNkY2DRiZv7xNcefMwfhwLfB"
           "3BFjB14u42DD32TkWsfareDdvNMd0Pnpr2IDhlFFKWQyk9cvWwI7rq8+"
           "7pQH9EdmELSEJJT8H4v6htblTJ0bWjwLZXPyH4pOHOJsLK5JaHzumK70wcqLu+"
           "iMf9yMxIE6lSFnSMf3GQ9tNgqDgAG92/"
           "VZzAEMBtHtzq4nDlcizFRD1TZoSjsP6AiRd9XAcWqVyWZgIbauYPnKFXXC+"
           "wMUKgMqSwZMRVbnhqIXsaKay4mjQ5E5Oo2YGBHCHHsIqKiJGDpdWO4JVv8Udg9wnEm"
           "tPls4H7bBJrR6PNZnuddhIjFUyQRgoZ/"
           "sOkoZKY7za+DX2UdVavu+r7UzxeKY5xVGiwRZWOBJ8uktlGu7Ie/"
           "F8bD0PZ8MxCNirtVp7krVAsCAwEAAQKCAgEAiOqYvJ97jZRWF7CiY5A6yswcm8DPlX"
           "tMdztx2Um7JnunU50r02aoVYOKFP1ik/LOy+uMN/"
           "M7SsvV2Kx714CSTVvEmPNYwdDLSPp+7zTD5yyyN5GVBWPeiImI0i1PdekOe53PfW+"
           "Ov5UQqa3nPX2rF7ZhX9MyUjtQOjoWjh3O4So4ZWV+D1lfkfQH+"
           "Vg9XosIYdrGmJybWxFSF0phTq/"
           "JqTPV7y47dykEdwlRGdUm3rHMa7nuANPUq9DddD+xy4ge9blSZFp/"
           "xLrvQ23xlwqrvQV2quyKH8ByA2QjQyBKTxLIV1aQTGqQszJTfrw+"
           "AONxkLwbsXLTvX+"
           "WPjNoY29YVq2XEWVHH9yEUWB4QLRlaIh6y4MafAvAdwjPiqpC7AgiweaHsxdAmn0vi"
           "pUov7DLQrhyWu318K3bjTyao2yjeUt0dG11t4Z2WAORhncuOeqTkhtkV3N+"
           "zRtncz3DYyy/xVy2bv1RY1ynz5ETjtRkNUmtd/TxxdM+CGrw6sIdA/"
           "IGvLSK3lLhSxqc/OHbY2FDhwSRsFiv4f09sevOsNI/"
           "u7DHtfF7utcHFImjeCMtTSMwFl1/wu/XwCMtXL8Ww0Qir7pl/"
           "oAUT+71tQ7iccgf9QBJ2tHBIeXNQvHXQpu1Dqd/"
           "9U9YDa9eHR3j4AJ1nuEflpSfNuVL3fB+"
           "ZCRQl5AojuECggEBAOme2HfMbtCKyJClTLTwtL8iB87+"
           "r8qG5DU0pweNVvbwGt0hj4uszQ0Zc1wy+"
           "p7C70rpxPCbl4msVPT3SJPoU15x8ZHr8LSNXx8pZak1EoQTX4PW0R1u2NI6S/"
           "4p0hE0fLEnNWD4mpBTkulIdr72NCYkSLm+DmM6TLOzuzdwa7ZEpdZ8LgIuSHLpHQ+"
           "RKo7cVU6QO7l75a99Lbv24yT5V1a+q+fivMbrkKLMeooDmZ/"
           "rAgJNT2AV3bUcEadZ6Q3ssmEUdbeXGcWyR62UIkSAXQOY7bgS8/"
           "3VBTN238FUjLCok7zZfPXkM/"
           "SSQQkNqb3pnu5UTJXu3YguPMjNBFw4hrECggEBAOlwPW9KgOWODI9sMxLs5WKdKLGv"
           "UWcCkdwW43Oa9RrbQIFz/RI9eHMcD3pyO2uHrGnvIR/gdgH/"
           "MOhw93CQwqNg5FbqChnNBWMeRs3UpfQZZIwK6zwWLdcPrQbTq0AzRoQgnXNL2ZDgaj"
           "G3qK3X1mTc5zk5rSXSU0TM0Vo7Ph2DdWM5zrQiMRyul3CxaXdyt2s3/"
           "Fy9T72En94jkVzq1Snee4KscMsfw20l+/"
           "yf+9nvZVrxckjByPXbFfYqhQu79GJhDYW+kiJ97SoaROgZpnU8P+"
           "bXDRpSCjbRgyjS6P+"
           "a60riLm9aQKgRJuvpDIi59cDvLTgsUFu909j9nGANrXsCggEBAJL4MDwlau1c8dI1e"
           "201JLA9a2YoSVUCyVxDdHTG0Rfmcy/8Jf77xb/g6/"
           "brx0jLX5jkOdYYPBwLQyMc0hbiiFgZHJztLm+XpGjJNUBet/"
           "hI68TYBpQaoOKYdChADPxv5aXZAbawjyQOv5Rs1u9JmgMIvovDae+"
           "4xHokrrmwysJOq3mkg5kxnrhTuMmLTJe1OyGCkWAd9qARAw2Fwt4G0pAqwaDOvm3x0"
           "fro4ooiu6mkQY5LLo1HFEQUlXKPmJL+/"
           "jUelvPZZgg+IOOtFmW7z0gtFBfCLKbaNTxsI1XPp/aXBuI+fGOAncLVo35bjxZ/"
           "e1F5zPhh3hIG7IxylbSihvECggEAZcD2FofcTbOwh+f7D1nR1z14GpO4IPW+"
           "zLfxZ1f5VT6TDJqZipTe5xYpLQ+UFRgrsHV68+Esr0wkbn2nbhq7rOm0Pqp/"
           "wqVansz4wgnaWN+ks2KjDSsBykfb/"
           "SjQ8ggnybpJx5MdgAhCYe+TiEuX+ZlM9OXOMrlOcKPy8aj3N3gw+"
           "BKfpmdtDvqPqqC6v0Vy7DqYN3iRxKOF6BeK6ny2dCy1m8080V3EGtS+"
           "1WyLJIE0pgKu9UX7KifYl9gK7IV+Tk/"
           "NzCm4m25Wrc6LkLx1kY60sBO3c2ylZSXITUg6JgjKXXbgFMu7eUcYtyZ7vzGIDRR0m"
           "pK+aG8DZAeQXVI49wKCAQEAhUjfDoZbF1je1vZMHxCNJICOARnyRq3QtMs/"
           "+dLAb6p59/Y7elqI80oMLS6xgYswFMHl7e/"
           "hYIu1fysXX0RVorJoNIwo03rPqnhKjabURpa4vCsAJodAYkYA0waknC1qJ3YkZgD2R"
           "W4RXF17PYnj0LHTN94/WbUA5X1FqarpnfGlfb/"
           "SWGQ1IGQ0IV0eRcviNWnN4MPVdh29oJk6oH4OHntXGs8b3jsdA9DZHOwOkzV3xUXoT"
           "DHlFBcVqMvTH/bZaweTaLw3+N8GlV6RlScwiRPh2YiJvEVCEVa4f4lrDunq/"
           "5G4Kt3tlWzluX2EM5zU5ZC/q2PqUdU8ysQforXL8g==\n-----END RSA PRIVATE "
           "KEY-----"}};
}

}  // virtual_network_helpers
}  // tests