// 
// Copyright (c) 2001-2003 The Trustees of Indiana University.  
//                         All rights reserved.
// Copyright (c) 1998-2001 University of Notre Dame. 
//                         All rights reserved.
// Copyright (c) 1994-1998 The Ohio State University.  
//                         All rights reserved.
// 
// This file is part of the XMPI software package.  For license
// information, see the LICENSE file in the top level directory of the
// XMPI source distribution.
// 
// $HEADER$
//
// $Id: xmpi_vendor.cc,v 1.4 2003/09/03 04:28:54 jsquyres Exp $
//
// xmpi_vendor
// 
// function: builds vendor pane
//

#include <stdio.h>
#include <stdlib.h>

#include <Xm/ArrowBG.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/List.h>
#include <Xm/PanedW.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/SelectioB.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>

#include <xmpi_vendor.h>
#include <xmpi_misc.h>
#include <xmpi_sys.h>
#include <xmpi_misc.h>
#include <xmpi_run.h>      // for xmpi_run_update()

#include <sfh.h>


extern xmpi_menu_item *run_options;
extern int run_options_length;
extern Widget *run_options_widgets;

extern xmpi_menu_item *schema_options;
extern int schema_options_length;
extern Widget *schema_options_widgets;

extern xmpi_node *nodes;
extern int nodes_length;

// Private functions...
// callback lists
static void vendor_schema_toggle_change_cb(Widget, XtPointer, XtPointer);
static void vendor_schema_textbox_change_cb(Widget, XtPointer, XtPointer);
static void vendor_run_toggle_change_cb(Widget, XtPointer, XtPointer);
static void vendor_run_textbox_change_cb(Widget, XtPointer, XtPointer);

static int draw_options_box(Widget, xmpi_menu_item*, int, Widget**,
			    XtCallbackProc, XtCallbackProc);

void
xmpi_vendor_build(Widget parent_w)
{
  Widget mgr_w;          // the vendor pane
  Widget framed_w;       // the lower portion, with the frame
  Widget schema_mgr_w;   // the grid for the schema stuff
  Widget run_mgr_w;      // the grid for the run stuff

  XmString xstr;         // label string.  Again, never need referenced.

  mgr_w = XtVaCreateWidget("vendor_mgr",
			   xmPanedWindowWidgetClass, parent_w,
			   XmNsashWidth, 1,
			   XmNsashHeight, 1,
			   XmNseparatorOn, False,
			   XmNleftAttachment, XmATTACH_POSITION,
			   XmNleftPosition, 2,
			   XmNleftOffset, 5,
			   XmNrightAttachment, XmATTACH_POSITION,
			   XmNrightPosition, 3,
			   XmNtopAttachment, XmATTACH_FORM,
			   XmNbottomAttachment, XmATTACH_FORM,
			   NULL);

  // create the top pane for the runtime options
  if (run_options_length != 0) {
    xstr = XmStringCreateSimple((char*) "Runtime Options");
    XtVaCreateManagedWidget("banner",
			    xmLabelWidgetClass, mgr_w,
			    XmNlabelString, xstr,
			    NULL);
    XmStringFree(xstr);
    
    framed_w = XtVaCreateManagedWidget("vendor_frame",
				       xmFrameWidgetClass, mgr_w,
				       XmNmarginHeight, 3,
				       XmNmarginWidth, 3,
				       XmNshadowThickness, 2,
				       XmNleftAttachment, XmATTACH_FORM,
				       XmNrightAttachment, XmATTACH_FORM,
				       NULL);
    
    run_mgr_w = XtVaCreateManagedWidget("run_mgr",
				 xmRowColumnWidgetClass, framed_w,
				 XmNisAligned, True,
				 XmNleftAttachment, XmATTACH_FORM,
				 XmNrightAttachment, XmATTACH_FORM,
				 NULL);
    if (draw_options_box(run_mgr_w, run_options, run_options_length,
			 &run_options_widgets, vendor_run_toggle_change_cb,
			 vendor_run_textbox_change_cb) != 0) {
      char error[] = "Runtime options window could not be drawn";
      xmpi_fail(error);
    }
  }

  // create the top pane for the schema options
  if (schema_options_length != 0) {
    xstr = XmStringCreateSimple((char*) "Schema Options");
    XtVaCreateManagedWidget("banner",
			    xmLabelWidgetClass, mgr_w,
			    XmNlabelString, xstr,
			  NULL);
    XmStringFree(xstr);
    
    framed_w = XtVaCreateManagedWidget("vendor_frame",
				       xmFrameWidgetClass, mgr_w,
				       XmNmarginHeight, 3,
				       XmNmarginWidth, 3,
				       XmNshadowThickness, 2,
				       XmNleftAttachment, XmATTACH_FORM,
				       XmNrightAttachment, XmATTACH_FORM,
				       NULL);
    
    schema_mgr_w = XtVaCreateManagedWidget("schema_mgr",
				    xmRowColumnWidgetClass, framed_w,
				    XmNisAligned, True,
				    XmNleftAttachment, XmATTACH_FORM,
				    XmNrightAttachment, XmATTACH_FORM,
				    NULL);

    if (draw_options_box(schema_mgr_w, schema_options, schema_options_length,
			 &schema_options_widgets,
			 vendor_schema_toggle_change_cb,
			 vendor_schema_textbox_change_cb) != 0) {
      char error[] = "Schema options window could not be drawn";
      xmpi_fail(error);
    }
  }

  xmpi_nosash(mgr_w);
  XtManageChild(mgr_w);
}


void 
vendor_schema_toggle_change_cb(Widget, XtPointer client_data, 
			       XtPointer call_data)
{
  int which = (int) client_data;
  XmToggleButtonCallbackStruct *state = 
    (XmToggleButtonCallbackStruct *) call_data;

  schema_options[which].current_val.xmpi_toggle = state->set;
  xmpi_run_update();
}

void
vendor_schema_textbox_change_cb(Widget ch_widget, XtPointer client_data,
				XtPointer)
{
  int which = (int) client_data;

  if (! (schema_options[which].current_val.xmpi_text == NULL))
    free(schema_options[which].current_val.xmpi_text);

  char* temp_string;

  temp_string = XmTextFieldGetString(ch_widget);
  if (strlen(temp_string) != 0)
    schema_options[which].current_val.xmpi_text = temp_string;
  else {
    schema_options[which].current_val.xmpi_text = NULL;
    free(temp_string);
  }
      
  xmpi_run_update();
}


void 
vendor_run_toggle_change_cb(Widget, XtPointer client_data, 
			       XtPointer call_data)
{
  int which = (int) client_data;
  XmToggleButtonCallbackStruct *state = 
    (XmToggleButtonCallbackStruct *) call_data;

  run_options[which].current_val.xmpi_toggle = state->set;
  xmpi_run_update();
}


void
vendor_run_textbox_change_cb(Widget ch_widget, XtPointer client_data,
				XtPointer)
{
  int which = (int) client_data;

  if (! (run_options[which].current_val.xmpi_text == NULL))
    free(run_options[which].current_val.xmpi_text);

  char* temp_string;

  temp_string = XmTextFieldGetString(ch_widget);
  if (strlen(temp_string) != 0)
    run_options[which].current_val.xmpi_text = temp_string;
  else {
    run_options[which].current_val.xmpi_text = NULL;
    free(temp_string);
  }
      
  xmpi_run_update();
}


int
draw_options_box(Widget mgr_w, xmpi_menu_item *options, int options_length,
		 Widget **options_widgets, XtCallbackProc toggle,
		 XtCallbackProc text)
{
  // stuff to deal with creating radio boxes
  Widget tmp_radio_w;
  int radio_id;

  Widget tmp_w;          // lbl widget.  don't need to keep this around 
  Widget h_pane;         // a widget to build txt boxes. dont need later

  XmString xstr;         // label string.  Again, never need referenced.

  // need to allocate space for widget array
  if (*options_widgets == 0) {
    // have not allocated yet
    *options_widgets = new Widget[options_length];
  } else {
    delete *options_widgets;
    *options_widgets = new Widget[options_length];
  }

  // at this point, go through the vendor list and 
  // build all the requested widgets.
  for (int i = 0 ; i < options_length ; ++i) {
    switch (options[i].xmpi_type) {
    case XMPI_MENU_LABEL:
      h_pane = XtVaCreateManagedWidget("h_pane",
				       xmFormWidgetClass, mgr_w,
				       XmNorientation, XmHORIZONTAL,
				       NULL);

      xstr = XmStringCreateSimple(options[i].xmpi_label);
      (*options_widgets)[i] = XtVaCreateManagedWidget("label",
					 xmLabelGadgetClass, h_pane,
					 XmNlabelString, xstr,
					 NULL);
      XmStringFree(xstr);
      break;

    case XMPI_MENU_TEXTBOX:
      // set the current to match the default value.
      if (options[i].default_val.xmpi_text != NULL) {
	options[i].current_val.xmpi_text = (char*) realloc(
		options[i].current_val.xmpi_text,
		sizeof(char) * (strlen(options[i].default_val.xmpi_text) + 1));

	strcpy(options[i].current_val.xmpi_text,
	       options[i].default_val.xmpi_text);
      } else {
	options[i].current_val.xmpi_text = NULL;
      }

      h_pane = XtVaCreateManagedWidget("h_pane",
				       xmFormWidgetClass, mgr_w,
				       XmNorientation, XmHORIZONTAL,
				       XmNleftAttachment, XmATTACH_FORM,
				       XmNrightAttachment, XmATTACH_FORM,
				       NULL);

      tmp_w = XtVaCreateManagedWidget(options[i].xmpi_label,
				      xmLabelGadgetClass, h_pane,
				      XmNalignment, XmALIGNMENT_BEGINNING,
				      XmNleftAttachment, XmATTACH_FORM,
				      NULL); 

      (*options_widgets)[i] = XtVaCreateManagedWidget("text",
				      xmTextFieldWidgetClass, h_pane,
				      XmNleftAttachment, XmATTACH_WIDGET,
				      XmNleftWidget, tmp_w,
				      XmNrightAttachment, XmATTACH_FORM,
				      NULL);
      


      XtAddCallback((*options_widgets)[i], XmNvalueChangedCallback,
		    text, (void*) i);
      
      if (options[i].current_val.xmpi_text != NULL)
	XmTextFieldInsert((*options_widgets)[i], 0, 
			  options[i].current_val.xmpi_text);

      break;

    case XMPI_MENU_TOGGLE:
      // make sure the current matches the default
      options[i].current_val.xmpi_toggle = options[i].default_val.xmpi_toggle;

      xstr = XmStringCreateSimple(options[i].xmpi_label);
      (*options_widgets)[i]= XtVaCreateManagedWidget("toggle",
			     xmToggleButtonWidgetClass, mgr_w,
			     XmNlabelString, xstr,
			     XmNset,(int) options[i].current_val.xmpi_toggle,
			     NULL);
      XmStringFree(xstr);

      XtAddCallback((*options_widgets)[i], XmNvalueChangedCallback,
		    toggle, (void*) i);

      break;

    case XMPI_MENU_RADIO:
      radio_id = options[i].xmpi_radio_id;
      
      // create the box to hold radio buttons.  this takes
      // care of the changing which is highlighted for us!
      tmp_radio_w = XtVaCreateManagedWidget("radio_box",
					 xmRowColumnWidgetClass, mgr_w,
					 XmNisAligned, True,
					 XmNradioBehavior, True,
					 NULL);
      
      for (; i < options_length ; ++i) {
	// make sure we are still on the right radio group and all
	if ((options[i].xmpi_type == XMPI_MENU_RADIO) && 
	      (options[i].xmpi_radio_id == radio_id))
	  break;

	// set the current to match what we are about to do
	options[i].current_val.xmpi_toggle = options[i].default_val.xmpi_toggle;
	
	xstr = XmStringCreateSimple(options[i].xmpi_label);
	(*options_widgets)[i] = XtVaCreateManagedWidget("radio",
			       xmToggleButtonWidgetClass, tmp_radio_w,
			       XmNlabelString, xstr,
			       XmNset, options[i].current_val.xmpi_toggle,
			       NULL);
	XmStringFree(xstr);

	XtAddCallback((*options_widgets)[i], XmNvalueChangedCallback,
		      toggle, (void*) i);
      }
      
      --i;  // since the for loop is about to increment...
      break;
    default:
      break;
    }

  }

  XtManageChild(mgr_w);

  return 0;
}


void
xmpi_vendor_update_run()
{
  for (int i = 0 ; i < run_options_length ; ++i) {
    switch (run_options[i].xmpi_type) {
    case XMPI_MENU_LABEL:
      break;
    case XMPI_MENU_TEXTBOX:
      if (run_options[i].current_val.xmpi_text != NULL) {
	XmTextFieldSetString(run_options_widgets[i],
			     run_options[i].current_val.xmpi_text);
      } else {
	XmTextFieldSetString(run_options_widgets[i], (char*) "");
      }
      break;
    case XMPI_MENU_TOGGLE:
    case XMPI_MENU_RADIO:
      XtVaSetValues(run_options_widgets[i],
		    XmNset,(int) run_options[i].current_val.xmpi_toggle,
		    NULL);
      break;
    default:
      ;
    }
  }
}


void
xmpi_vendor_update_schema()
{
  for (int i = 0 ; i < schema_options_length ; ++i) {
    switch (schema_options[i].xmpi_type) {
    case XMPI_MENU_LABEL:
      break;
    case XMPI_MENU_TEXTBOX:
      if (schema_options[i].current_val.xmpi_text != NULL) {
	XmTextFieldSetString(schema_options_widgets[i],
			     schema_options[i].current_val.xmpi_text);
      } else {
	XmTextFieldSetString(schema_options_widgets[i], (char*) "");
      }
      break;
    case XMPI_MENU_TOGGLE:
    case XMPI_MENU_RADIO:
      XtVaSetValues(schema_options_widgets[i],
		    XmNset,(int) schema_options[i].current_val.xmpi_toggle,
		    NULL);
      break;
    default:
      ;
    }
  }
}


void
xmpi_vendor_update_schema_default()
{
  char *blank = strdup("");
  for (int i = 0 ; i < schema_options_length ; ++i) {
    switch (schema_options[i].xmpi_type) {
    case XMPI_MENU_LABEL:
      break;
    case XMPI_MENU_TEXTBOX:
      if (schema_options[i].default_val.xmpi_text != NULL)
	XmTextFieldSetString(schema_options_widgets[i],
			     schema_options[i].default_val.xmpi_text);
      else
	XmTextFieldSetString(schema_options_widgets[i], blank);
      break;
    case XMPI_MENU_TOGGLE:
    case XMPI_MENU_RADIO:
      XtVaSetValues(schema_options_widgets[i],
		    XmNset,(int) schema_options[i].default_val.xmpi_toggle,
		    NULL);
      break;
    default:
      ;
    }
  }
  
}


void
xmpi_vendor_cleanup()
{
  if (run_options_length != 0) {
    xmpi_vendor_clean_options(run_options, run_options_length, true);
    free(run_options);
    run_options_length = 0;
  }

  if (schema_options_length != 0) {
    xmpi_vendor_clean_options(schema_options, schema_options_length, true);
    free(schema_options);
    schema_options_length = 0;
  }

  if (nodes_length != 0) {
    for (int i = 0 ; i < nodes_length ; ++i)
      if (nodes[i].name != NULL) {
	free(nodes[i].name);
      }
    free(nodes);
  }
}

 
void
xmpi_vendor_clean_options(xmpi_menu_item *options, 
			  int options_length, bool all)
{
  for (int i = 0 ; i < options_length ; ++i) {
    if (options[i].xmpi_type == XMPI_MENU_TEXTBOX) {
      if (options[i].current_val.xmpi_text != NULL) {
	free(options[i].current_val.xmpi_text);
	options[i].current_val.xmpi_text = NULL;
      }
      
      if (options[i].default_val.xmpi_text != NULL && all) {
	free(options[i].default_val.xmpi_text);
	options[i].default_val.xmpi_text = NULL;
      }
    }
    
    if (options[i].xmpi_label != NULL && all) {
      free(options[i].xmpi_label);
      options[i].xmpi_label = NULL;
    }
  }
}


